import { jsx as _jsx, Fragment as _Fragment } from "react/jsx-runtime";
/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import React from 'react';
import { Box, Text } from 'ink';
import { DiffRenderer } from './DiffRenderer.js';
import { MarkdownDisplay } from '../../utils/MarkdownDisplay.js';
import { AnsiOutputText } from '../AnsiOutput.js';
import { MaxSizedBox } from '../shared/MaxSizedBox.js';
import { theme } from '../../semantic-colors.js';
import { useUIState } from '../../contexts/UIStateContext.js';
import { useAlternateBuffer } from '../../hooks/useAlternateBuffer.js';
const STATIC_HEIGHT = 1;
const RESERVED_LINE_COUNT = 5; // for tool name, status, padding etc.
const MIN_LINES_SHOWN = 2; // show at least this many lines
// Large threshold to ensure we don't cause performance issues for very large
// outputs that will get truncated further MaxSizedBox anyway.
const MAXIMUM_RESULT_DISPLAY_CHARACTERS = 20000;
export const ToolResultDisplay = ({ resultDisplay, availableTerminalHeight, terminalWidth, renderOutputAsMarkdown = true, }) => {
    const { renderMarkdown } = useUIState();
    const isAlternateBuffer = useAlternateBuffer();
    const availableHeight = availableTerminalHeight
        ? Math.max(availableTerminalHeight - STATIC_HEIGHT - RESERVED_LINE_COUNT, MIN_LINES_SHOWN + 1)
        : undefined;
    // Long tool call response in MarkdownDisplay doesn't respect availableTerminalHeight properly,
    // so if we aren't using alternate buffer mode, we're forcing it to not render as markdown when the response is too long, it will fallback
    // to render as plain text, which is contained within the terminal using MaxSizedBox
    if (availableHeight && !isAlternateBuffer) {
        renderOutputAsMarkdown = false;
    }
    const combinedPaddingAndBorderWidth = 4;
    const childWidth = terminalWidth - combinedPaddingAndBorderWidth;
    const truncatedResultDisplay = React.useMemo(() => {
        if (typeof resultDisplay === 'string') {
            if (resultDisplay.length > MAXIMUM_RESULT_DISPLAY_CHARACTERS) {
                return '...' + resultDisplay.slice(-MAXIMUM_RESULT_DISPLAY_CHARACTERS);
            }
        }
        return resultDisplay;
    }, [resultDisplay]);
    if (!truncatedResultDisplay)
        return null;
    return (_jsx(Box, { width: childWidth, flexDirection: "column", children: _jsx(Box, { flexDirection: "column", children: typeof truncatedResultDisplay === 'string' &&
                renderOutputAsMarkdown ? (_jsx(Box, { flexDirection: "column", children: _jsx(MarkdownDisplay, { text: truncatedResultDisplay, terminalWidth: childWidth, renderMarkdown: renderMarkdown, isPending: false }) })) : typeof truncatedResultDisplay === 'string' &&
                !renderOutputAsMarkdown ? (isAlternateBuffer ? (_jsx(Box, { flexDirection: "column", width: childWidth, children: _jsx(Text, { wrap: "wrap", color: theme.text.primary, children: truncatedResultDisplay }) })) : (_jsx(MaxSizedBox, { maxHeight: availableHeight, maxWidth: childWidth, children: _jsx(Box, { children: _jsx(Text, { wrap: "wrap", color: theme.text.primary, children: truncatedResultDisplay }) }) }))) : typeof truncatedResultDisplay === 'object' &&
                'fileDiff' in truncatedResultDisplay ? (_jsx(DiffRenderer, { diffContent: truncatedResultDisplay.fileDiff, filename: truncatedResultDisplay.fileName, availableTerminalHeight: availableHeight, terminalWidth: childWidth })) : typeof truncatedResultDisplay === 'object' &&
                'todos' in truncatedResultDisplay ? (
            // display nothing, as the TodoTray will handle rendering todos
            _jsx(_Fragment, {})) : (_jsx(AnsiOutputText, { data: truncatedResultDisplay, availableTerminalHeight: availableHeight, width: childWidth })) }) }));
};
//# sourceMappingURL=ToolResultDisplay.js.map