# -*- coding: utf-8 -*- #
# Copyright 2025 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Command to clear Artifact Registry Platform Logs configuration."""

from googlecloudsdk.api_lib.util import apis
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.artifacts import flags
from googlecloudsdk.command_lib.artifacts import platformlogs_util
from googlecloudsdk.core import log


@base.ReleaseTracks(base.ReleaseTrack.GA)
@base.Hidden
@base.UniverseCompatible
class Clear(base.UpdateCommand):
  """Clear the Platform Logs configuration."""

  api_version = 'v1'

  detailed_help = {
      'DESCRIPTION': '{description}',
      'EXAMPLES': """
          To clear the Platform Logs configuration for the project 'my-project' in us-west1 (move project to default state):

            $ {command} --project=my-project --location=us-west1

          To clear the Platform Logs configuration for the repository 'my-repo' in us-west1 (fall back to project-level logging configuration):

            $ {command} --project=my-project --location=us-west1 --repository=my-repo
          """,
  }

  @staticmethod
  def Args(parser):
    """Set up arguments for this command.

    Args:
      parser: An argparse.ArgumentPaser.
    """
    flags.GetPlainRepoFlag().AddToParser(parser)
    flags.GetPlainLocationFlag().AddToParser(parser)

  def Run(self, args):
    """Run the clear command."""
    client = apis.GetClientInstance('artifactregistry', self.api_version)
    messages = client.MESSAGES_MODULE
    platform_logs_config = messages.PlatformLogsConfig()

    response = platformlogs_util.UpdatePlatformLogsConfig(
        args, client, messages, platform_logs_config
    )
    log.status.Print(
        'Cleared Platform Logs Config for [{}].'.format(response.name)
    )
    return response
