# -*- coding: utf-8 -*- #
# Copyright 2025 Google LLC. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""gcloud emulators spanner notices command."""

from __future__ import absolute_import
from __future__ import division
from __future__ import unicode_literals

import argparse

from googlecloudsdk.calliope import arg_parsers
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.emulators import spanner_util


@base.ReleaseTracks(base.ReleaseTrack.ALPHA, base.ReleaseTrack.BETA,
                    base.ReleaseTrack.GA)
@base.UniverseCompatible
class Notices(base.Command):
  """Print third-party notices for the local Cloud Spanner emulator.
  """

  detailed_help = {
      'EXAMPLES':
          """\
          To print third-party notices for the local Cloud Spanner emulator, run:

            $ {command}
          """,
  }

  @staticmethod
  def Args(parser):
    parser.add_argument(
        '--use-docker',
        required=False,
        action='store_true',
        help='Use the Cloud Spanner emulator docker image even if the platform '
        'has a native binary available in the gcloud CLI. Currently we only '
        'provide a native binary for Linux. For other systems, you must '
        'install Docker for your platform before starting the emulator.')

  def Run(self, args):
    # Construct a new args for spanner_util.Start. No need to parse arguments
    # for this command.
    start_args = argparse.Namespace()
    # These attributes are needed by spanner_util._BuildStartArgs*
    start_args.host_port = arg_parsers.HostPort('localhost', '9010')
    start_args.rest_port = 9020
    start_args.use_docker = args.use_docker
    start_args.enable_fault_injection = False

    # This is the main point of this command.
    start_args.print_notices = True
    spanner_util.Start(start_args)
