import typing
import warnings
from abc import abstractmethod
from collections.abc import Callable, Hashable
from typing import Optional

from pytensor.graph.basic import Apply, Constant
from pytensor.graph.utils import MethodNotDefined


if typing.TYPE_CHECKING:
    from pytensor.link.c.params_type import Params, ParamsType


class CLinkerObject:
    """Standard methods for an `Op` or `Type` used with the `CLinker`."""

    def c_headers(self, **kwargs) -> list[str]:
        """Return a list of header files required by code returned by this class.

        These strings will be prefixed with ``#include`` and inserted at the
        beginning of the C source code.

        Strings in this list that start neither with ``<`` nor ``"`` will be
        enclosed in double-quotes.

        Examples
        --------

        .. code-block:: python

            def c_headers(self, **kwargs):
                return ["<iostream>", "<math.h>", "/full/path/to/header.h"]


        """
        return []

    def c_header_dirs(self, **kwargs) -> list[str]:
        """Return a list of header search paths required by code returned by this class.

        Provides search paths for headers, in addition to those in any relevant
        environment variables.

        .. note::

            For Unix compilers, these are the things that get ``-I`` prefixed
            in the compiler command line arguments.


        Examples
        --------

        .. code-block:: python

            def c_header_dirs(self, **kwargs):
                return ["/usr/local/include", "/opt/weirdpath/src/include"]

        """
        return []

    def c_libraries(self, **kwargs) -> list[str]:
        """Return a list of libraries required by code returned by this class.

        The compiler will search the directories specified by the environment
        variable ``LD_LIBRARY_PATH`` in addition to any returned by
        :meth:`CLinkerOp.c_lib_dirs`.

        .. note::

            For Unix compilers, these are the things that get ``-l`` prefixed
            in the compiler command line arguments.


        Examples
        --------

        .. code-block:: python

            def c_libraries(self, **kwargs):
                return ['gsl', 'gslcblas', 'm', 'fftw3', 'g2c'].

        """
        return []

    def c_lib_dirs(self, **kwargs) -> list[str]:
        """Return a list of library search paths required by code returned by this class.

        Provides search paths for libraries, in addition to those in any
        relevant environment variables (e.g. ``LD_LIBRARY_PATH``).

        .. note::

            For Unix compilers, these are the things that get ``-L`` prefixed
            in the compiler command line arguments.


        Examples
        --------

        .. code-block:: python

            def c_lib_dirs(self, **kwargs):
                return ['/usr/local/lib', '/opt/weirdpath/build/libs'].

        """
        return []

    def c_support_code(self, **kwargs) -> str:
        """Return utility code for use by a `Variable` or `Op`.

        This is included at global scope prior to the rest of the code for this class.

        Question: How many times will this support code be emitted for a graph
        with many instances of the same type?

        Returns
        -------
        str

        """
        return ""

    def c_compile_args(self, **kwargs) -> list[str]:
        """Return a list of recommended compile arguments for code returned by other methods in this class.

        Compiler arguments related to headers, libraries and search paths
        should be provided via the functions `c_headers`, `c_libraries`,
        `c_header_dirs`, and `c_lib_dirs`.

        Examples
        --------

        .. code-block:: python

            def c_compile_args(self, **kwargs):
                return ["-ffast-math"]

        """
        return []

    def c_no_compile_args(self, **kwargs) -> list[str]:
        """Return a list of incompatible ``gcc`` compiler arguments.

        We will remove those arguments from the command line of ``gcc``. So if
        another `Op` adds a compile arg in the graph that is incompatible
        with this `Op`, the incompatible arg will not be used.

        This is used, for instance, to remove ``-ffast-math``.

        """
        return []

    def c_init_code(self, **kwargs) -> list[str]:
        """Return a list of code snippets to be inserted in module initialization."""
        return []

    def c_code_cache_version(self) -> tuple[Hashable, ...]:
        """Return a tuple of integers indicating the version of this `Op`.

        An empty tuple indicates an "unversioned" `Op` that will not be cached
        between processes.

        The cache mechanism may erase cached modules that have been superseded
        by newer versions. See `ModuleCache` for details.

        See Also
        --------
        c_code_cache_version_apply()

        """
        warnings.warn(
            f"{type(self)} specifies no C code cache version and will not be cached."
        )
        return ()


class CLinkerOp(CLinkerObject):
    """Interface definition for `Op` subclasses compiled by `CLinker`."""

    params_type: Optional["ParamsType"] = None

    @abstractmethod
    def c_code(
        self,
        node: Apply,
        name: str,
        inputs: list[str],
        outputs: list[str],
        sub: dict[str, str],
    ) -> str:
        """Return the C implementation of an ``Op``.

        Returns C code that does the computation associated to this ``Op``,
        given names for the inputs and outputs.

        Parameters
        ----------
        node : Apply instance
            The node for which we are compiling the current C code.
            The same ``Op`` may be used in more than one node.
        name : str
            A name that is automatically assigned and guaranteed to be
            unique.
        inputs : list of strings
            There is a string for each input of the function, and the
            string is the name of a C variable pointing to that input.
            The type of the variable depends on the declared type of
            the input.  There is a corresponding python variable that
            can be accessed by prepending ``"py_"`` to the name in the
            list.
        outputs : list of strings
            Each string is the name of a C variable where the `Op` should
            store its output.  The type depends on the declared type of
            the output.  There is a corresponding Python variable that
            can be accessed by prepending ``"py_"`` to the name in the
            list.  In some cases the outputs will be preallocated and
            the value of the variable may be pre-filled.  The value for
            an unallocated output is type-dependent.
        sub : dict of strings
            Extra symbols defined in `CLinker` sub symbols (such as ``'fail'``).

        """
        raise NotImplementedError()

    def c_code_cache_version_apply(self, node: Apply) -> tuple[Hashable, ...]:
        """Return a tuple of integers indicating the version of this `Op`.

        An empty tuple indicates an "unversioned" `Op` that will not be
        cached between processes.

        The cache mechanism may erase cached modules that have been
        superseded by newer versions.  See `ModuleCache` for details.

        See Also
        --------
        c_code_cache_version

        Notes
        -----
            This function overrides `c_code_cache_version` unless it explicitly
            calls `c_code_cache_version`. The default implementation simply
            calls `c_code_cache_version` and ignores the `node` argument.

        """
        return self.c_code_cache_version()

    def c_code_cleanup(
        self,
        node: Apply,
        name: str,
        inputs: list[str],
        outputs: list[str],
        sub: dict[str, str],
    ) -> str:
        """Return C code to run after :meth:`CLinkerOp.c_code`, whether it failed or not.

        This is a convenient place to clean up things allocated by :meth:`CLinkerOp.c_code`.

        Parameters
        ----------
        node : Apply
            WRITEME
        name : str
            A name that is automatically assigned and guaranteed to be
            unique.
        inputs : list of str
            There is a string for each input of the function, and the
            string is the name of a C variable pointing to that input.
            The type of the variable depends on the declared type of
            the input. There is a corresponding Python variable that
            can be accessed by prepending ``"py_"`` to the name in the
            list.
        outputs : list of str
            Each string is the name of a C variable corresponding to
            one of the outputs of the `Op`. The type depends on the
            declared type of the output. There is a corresponding
            Python variable that can be accessed by prepending ``"py_"`` to
            the name in the list.
        sub : dict of str
            Extra symbols defined in `CLinker` sub symbols (such as ``'fail'``).

        """
        return ""

    def c_support_code_apply(self, node: Apply, name: str) -> str:
        """Return `Apply`-specialized utility code for use by an `Op` that will be inserted at global scope.

        Parameters
        ----------
        node : Apply
            The node in the graph being compiled.
        name : str
            A string or number that serves to uniquely identify this node.
            Symbol names defined by this support code should include the name,
            so that they can be called from the :meth:`CLinkerOp.c_code`, and so that
            they do not cause name collisions.

        Notes
        -----
        This function is called in addition to :meth:`CLinkerObject.c_support_code`
        and will supplement whatever is returned from there.

        """
        return ""

    def c_init_code_apply(self, node: Apply, name: str) -> str:
        """Return a code string specific to the `Apply` to be inserted in the module initialization code.

        Parameters
        ----------
        node
            An `Apply` instance in the graph being compiled
        name : str
            A string or number that serves to uniquely identify this node.
            Symbol names defined by this support code should include the name,
            so that they can be called from :meth:`CLinkerOp.c_code`, and so
            that they do not cause name collisions.

        Notes
        -----
        This function is called in addition to
        :meth:`CLinkerObject.c_init_code` and will supplement whatever is
        returned from there.

        """
        return ""

    def c_init_code_struct(self, node: Apply, name, sub) -> str:
        """Return an `Apply`-specific code string to be inserted in the struct initialization code.

        Parameters
        ----------
        node : Apply
            The node in the graph being compiled.
        name : str
            A unique name to distinguish variables from those of other nodes.
        sub : dict of str
            A dictionary of values to substitute in the code.
            Most notably it contains a ``'fail'`` entry that you should place
            in your code after setting a Python exception to indicate an error.

        """
        return ""

    def c_support_code_struct(self, node: Apply, name: str) -> str:
        """Return `Apply`-specific utility code for use by an `Op` that will be inserted at struct scope.

        Parameters
        ----------
        node : Apply
            The node in the graph being compiled
        name : str
            A unique name to distinguish you variables from those of other
            nodes.

        """
        return ""

    def c_cleanup_code_struct(self, node: Apply, name: str) -> str:
        """Return an `Apply`-specific code string to be inserted in the struct cleanup code.

        Parameters
        ----------
        node : Apply
            The node in the graph being compiled
        name : str
            A unique name to distinguish variables from those of other nodes.

        """
        return ""

    def get_params(self, node: Apply) -> "Params":
        """Try to get parameters for the `Op` when :attr:`Op.params_type` is set to a `ParamsType`."""
        if self.params_type is not None:
            wrapper = self.params_type
            if not all(hasattr(self, field) for field in wrapper.fields):
                # Let's print missing attributes for debugging.
                not_found = tuple(
                    field for field in wrapper.fields if not hasattr(self, field)
                )
                raise AttributeError(
                    f"{type(self).__name__}: missing attributes {not_found} for ParamsType."
                )
            # ParamsType.get_params() will apply filtering to attributes.
            return self.params_type.get_params(self)
        raise MethodNotDefined("get_params")


class CLinkerType(CLinkerObject):
    r"""Interface specification for `Type`\s that can be arguments to a `CLinkerOp`.

    A `CLinkerType` instance is mainly responsible  for providing the C code that
    interfaces python objects with a C `CLinkerOp` implementation.

    """

    @abstractmethod
    def c_declare(
        self, name: str, sub: dict[str, str], check_input: bool = True
    ) -> str:
        """Return C code to declare variables that will be instantiated by :meth:`CLinkerType.c_extract`.

        Parameters
        ----------
        name
            The name of the ``PyObject *`` pointer that will the value for this
            `Type`.
        sub
            A dictionary of special codes.  Most importantly
            ``sub['fail']``. See `CLinker` for more info on ``sub`` and
            ``fail``.

        Notes
        -----
        It is important to include the `name` inside of variables which
        are declared here, so that name collisions do not occur in the
        source file that is generated.

        The variable called `name` is not necessarily defined yet
        where this code is inserted. This code might be inserted to
        create class variables for example, whereas the variable `name`
        might only exist inside certain functions in that class.

        TODO: Why should variable declaration fail?  Is it even allowed to?

        Examples
        --------

        .. code-block: python

            def c_declare(self, name, sub, check_input=True):
                return "PyObject ** addr_of_%(name)s;"

        """

    @abstractmethod
    def c_init(self, name: str, sub: dict[str, str]) -> str:
        """Return C code to initialize the variables that were declared by :meth:`CLinkerType.c_declare`.

        Notes
        -----
        The variable called `name` is not necessarily defined yet
        where this code is inserted. This code might be inserted in a
        class constructor for example, whereas the variable `name`
        might only exist inside certain functions in that class.

        TODO: Why should variable initialization fail?  Is it even allowed to?

        Examples
        --------

        .. code-block: python

            def c_init(self, name, sub):
                return "addr_of_%(name)s = NULL;"

        """

    @abstractmethod
    def c_extract(
        self, name: str, sub: dict[str, str], check_input: bool = True, **kwargs
    ) -> str:
        r"""Return C code to extract a ``PyObject *`` instance.

        The code returned from this function must be templated using
        ``%(name)s``, representing the name that the caller wants to
        call this `Variable`. The Python object ``self.data`` is in a
        variable called ``"py_%(name)s"`` and this code must set the
        variables declared by :meth:`CLinkerType.c_declare` to something
        representative of ``py_%(name)``\s. If the data is improper, set an
        appropriate exception and insert ``"%(fail)s"``.

        TODO: Point out that template filling (via sub) is now performed
        by this function. --jpt

        Parameters
        ----------
        name
            The name of the ``PyObject *`` pointer that will store the value
            for this type.
        sub
            A dictionary of special codes. Most importantly
            ``sub['fail']``. See `CLinker` for more info on ``sub`` and
            ``fail``.

        Examples
        --------

        .. code-block: python

            def c_extract(self, name, sub, check_input=True, **kwargs):
                return "if (py_%(name)s == Py_None)" + \\\
                            addr_of_%(name)s = &py_%(name)s;" + \\\
                    "else" + \\\
                    { PyErr_SetString(PyExc_ValueError, \\\
                            'was expecting None'); %(fail)s;}"

        """

    @abstractmethod
    def c_sync(self, name: str, sub: dict[str, str]) -> str:
        """Return C code to pack C types back into a ``PyObject``.

        The code returned from this function must be templated using
        ``"%(name)s"``, representing the name that the caller wants to
        call this `Variable`. The returned code may set ``"py_%(name)s"``
        to a ``PyObject*`` and that ``PyObject*`` will be accessible from
        Python via ``variable.data``. Do not forget to adjust reference
        counts if ``"py_%(name)s"`` is changed from its original value.

        Parameters
        ----------
        name
            WRITEME
        sub
            WRITEME

        """

    def c_element_type(self) -> str:
        """Return the name of the primitive C type of items into variables handled by this type.

        e.g:

         - For ``TensorType(dtype='int64', ...)``: should return ``"npy_int64"``.
        """
        return ""

    def c_is_simple(self) -> bool:
        """Return ``True`` for small or builtin C types.

        A hint to tell the compiler that this type is a builtin C type or a
        small struct and that its memory footprint is negligible. Simple
        objects may be passed on the stack.

        """
        return False

    def c_literal(self, data: Constant) -> str:
        """Provide a C literal string value for the specified `data`.

        Parameters
        ----------
        data
            The data to be converted into a C literal string.

        """
        return ""

    def c_extract_out(
        self, name: str, sub: dict[str, str], check_input: bool = True, **kwargs
    ) -> str:
        """Return C code to extract a ``PyObject *`` instance.

        Unlike :math:`CLinkerType.c_extract`, :meth:`CLinkerType.c_extract_out` has to
        accept ``Py_None``, meaning that the variable should be left
        uninitialized.

        """
        c_init_code = self.c_init(name, sub)
        c_extract_code = self.c_extract(name, sub, check_input)
        return f"""
        if (py_{name} == Py_None)
        {{
            {c_init_code}
        }}
        else
        {{
            {c_extract_code}
        }}
        """

    def c_cleanup(self, name: str, sub: dict[str, str]) -> str:
        """Return C code to clean up after :meth:`CLinkerType.c_extract`.

        This returns C code that should deallocate whatever
        :meth:`CLinkerType.c_extract` allocated or decrease the reference counts. Do
        not decrease ``py_%(name)s``'s reference count.

        Parameters
        ----------
        name : str
            WRITEME
        sub : dict of str
            WRITEME

        """
        return ""

    def c_code_cache_version(self) -> tuple[int, ...]:
        """Return a tuple of integers indicating the version of this type.

        An empty tuple indicates an "unversioned" type that will not
        be cached between processes.

        The cache mechanism may erase cached modules that have been
        superseded by newer versions. See `ModuleCache` for details.

        """
        return ()


class HideC(CLinkerOp):
    def __hide(*args):
        raise MethodNotDefined()

    c_code: Callable = __hide
    c_code_cleanup: Callable = __hide

    c_headers: Callable = __hide
    c_header_dirs: Callable = __hide
    c_libraries: Callable = __hide
    c_lib_dirs: Callable = __hide

    c_support_code: Callable = __hide
    c_support_code_apply: Callable = __hide

    c_compile_args: Callable = __hide
    c_no_compile_args: Callable = __hide
    c_init_code: Callable = __hide
    c_init_code_apply: Callable = __hide

    c_init_code_struct: Callable = __hide
    c_support_code_struct: Callable = __hide
    c_cleanup_code_struct: Callable = __hide

    def c_code_cache_version(self):
        return ()

    def c_code_cache_version_apply(self, node):
        return self.c_code_cache_version()
